/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <libepubgen/libepubgen.h>

#include <libetonyek/libetonyek.h>

#include "EpubPackage.hxx"
#include "UsageHelper.hxx"

#define TOOLNAME "key2epub"

using libetonyek::EtonyekDocument;

class EpubOutputFileHelper
{
public:
	EpubOutputFileHelper(const char *const outFileName) :
		m_fileName(outFileName)
	{
	}

	bool convertDocument(librevenge::RVNGInputStream &input)
	{
		EpubPackage package(m_fileName);
		libepubgen::EPUBPresentationGenerator collector(&package);

		return EtonyekDocument::parse(&input, &collector);
	}

	EtonyekDocument::Confidence isSupportedFormat(librevenge::RVNGInputStream &input, EtonyekDocument::Type *type = 0)
	{
		return EtonyekDocument::isSupported(&input, type);
	}

private:
	std::string m_fileName;
};

int printUsage(const char *name)
{
	UsageHelper usage(name, "converts Apple Keynote documents to ePub.", "[OPTIONS] INPUT OUTPUT");
	return usage.printUsage();
}

int main(int argc, char *argv[])
{
	char *szInputFile = 0;
	char *szOutFile = 0;

	for (int i = 1; i < argc; i++)
	{
		if (strcmp(argv[i], "--version")==0)
			return UsageHelper::printVersion(TOOLNAME);
		else if (strcmp(argv[i], "--help")==0)
			return printUsage(TOOLNAME);
		else if (!szInputFile && strncmp(argv[i], "--", 2))
			szInputFile = argv[i];
		else if (szInputFile && !szOutFile && strncmp(argv[i], "--", 2))
			szOutFile = argv[i];
		else
			return printUsage(TOOLNAME);
	}

	if (!szInputFile || !szOutFile)
		return printUsage(TOOLNAME);

	shared_ptr<librevenge::RVNGInputStream> input;
	if (librevenge::RVNGDirectoryStream::isDirectory(szInputFile))
		input.reset(new librevenge::RVNGDirectoryStream(szInputFile));
	else
		input.reset(new librevenge::RVNGFileStream(szInputFile));

	EpubOutputFileHelper helper(szOutFile);

	EtonyekDocument::Type type = EtonyekDocument::TYPE_UNKNOWN;
	const EtonyekDocument::Confidence confidence = helper.isSupportedFormat(*input, &type);
	if ((EtonyekDocument::CONFIDENCE_NONE == confidence) || (EtonyekDocument::TYPE_KEYNOTE != type))
	{
		fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Pages Document.\n");
		return 1;
	}

	if (EtonyekDocument::CONFIDENCE_SUPPORTED_PART == confidence)
	{
		input.reset(librevenge::RVNGDirectoryStream::createForParent(szInputFile));

		if (EtonyekDocument::CONFIDENCE_EXCELLENT != helper.isSupportedFormat(*input))
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Pages Document.\n");
			return 1;
		}
	}

	if (!helper.convertDocument(*input))
	{
		fprintf(stderr, "ERROR : Couldn't convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
