/*  Lziprecover - Data recovery tool for the lzip format
    Copyright (C) 2009-2016 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#define _FILE_OFFSET_BITS 64

#include <algorithm>
#include <cerrno>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <string>
#include <vector>
#include <stdint.h>
#include <unistd.h>
#include <sys/stat.h>

#include "lzip.h"
#include "decoder.h"
#include "block.h"
#include "file_index.h"


namespace {

int decompress_member( const int infd, const Pretty_print & pp,
                       const unsigned long long mpos,
                       const unsigned long long outskip,
                       const unsigned long long outend )
  {
  Range_decoder rdec( infd );
  File_header header;
  rdec.read_data( header.data, File_header::size );
  if( rdec.finished() )			// End Of File
    { pp( "File ends unexpectedly at member header." ); return 2; }
  if( !verify_header( header, pp ) ) return 2;
  const unsigned dictionary_size = header.dictionary_size();
  if( !isvalid_ds( dictionary_size ) )
    { pp( "Invalid dictionary size in member header." ); return 2; }

  if( pp.verbosity() >= 2 ) { pp(); show_header( dictionary_size ); }

  LZ_decoder decoder( rdec, dictionary_size, outfd, outskip, outend );
  const int result = decoder.decode_member( pp );
  if( result != 0 )
    {
    if( pp.verbosity() >= 0 && result <= 2 )
      {
      pp();
      std::fprintf( stderr, "%s at pos %llu\n", ( result == 2 ) ?
                    "File ends unexpectedly" : "Decoder error",
                    mpos + rdec.member_position() );
      }
    return 2;
    }
  if( pp.verbosity() >= 2 ) std::fputs( "done\n", stderr );
  return 0;
  }


int list_file( const char * const input_filename, const Pretty_print & pp )
  {
  struct stat in_stats;				// not used
  const int infd = open_instream( input_filename, &in_stats, true, true );
  if( infd < 0 ) return 1;

  const File_index file_index( infd );
  close( infd );
  if( file_index.retval() != 0 )
    { pp( file_index.error().c_str() ); return file_index.retval(); }

  if( pp.verbosity() >= 0 )
    {
    const unsigned long long data_size = file_index.data_end();
    const unsigned long long file_size = file_index.file_end();
    unsigned dictionary_size = 0;
    for( long i = 0; i < file_index.members(); ++i )
      dictionary_size =
        std::max( dictionary_size, file_index.dictionary_size( i ) );
    pp( 0, stdout );
    show_header( dictionary_size, 1 );
    if( data_size > 0 && file_size > 0 )
      std::printf( "%6.3f:1, %6.3f bits/byte, %5.2f%% saved.  ",
                   (double)data_size / file_size,
                   ( 8.0 * file_size ) / data_size,
                   100.0 * ( 1.0 - ( (double)file_size / data_size ) ) );
    std::printf( "decompressed size %9llu, compressed size %8llu.\n",
                  data_size, file_size );

    if( pp.verbosity() >= 1 && file_index.members() > 1 )
      {
      std::printf( "    Total members in file = %ld\n", file_index.members() );
      if( pp.verbosity() >= 2 )
        for( long i = 0; i < file_index.members(); ++i )
          {
          const Block & db = file_index.dblock( i );
          const Block & mb = file_index.mblock( i );
          std::printf( "    Member %3ld   data pos %9llu   data size %7llu   "
                       "member pos %9llu   member size %7llu.\n", i + 1,
                       db.pos(), db.size(), mb.pos(), mb.size() );
          }
      }
    const long long trailing_size = file_index.file_size() - file_index.file_end();
    if( pp.verbosity() >= 1 && trailing_size > 0 )
      std::printf( "    %lld bytes of trailing data at end of file.\n",
                   trailing_size );
    }
  return 0;
  }

} // end namespace


const char * format_num( unsigned long long num,
                         unsigned long long limit,
                         const int set_prefix )
  {
  const char * const si_prefix[8] =
    { "k", "M", "G", "T", "P", "E", "Z", "Y" };
  const char * const binary_prefix[8] =
    { "Ki", "Mi", "Gi", "Ti", "Pi", "Ei", "Zi", "Yi" };
  enum { buffers = 8, bufsize = 32 };
  static char buffer[buffers][bufsize];	// circle of static buffers for printf
  static int current = 0;
  static bool si = true;

  if( set_prefix ) si = ( set_prefix > 0 );
  const unsigned factor = ( si ? 1000 : 1024 );
  char * const buf = buffer[current++]; current %= buffers;
  const char * const * prefix = ( si ? si_prefix : binary_prefix );
  const char * p = "";
  bool exact = ( num % factor == 0 );

  for( int i = 0; i < 8 && ( num > limit || ( exact && num >= factor ) ); ++i )
    { num /= factor; if( num % factor != 0 ) exact = false; p = prefix[i]; }
  snprintf( buf, bufsize, "%llu %s", num, p );
  return buf;
  }


bool safe_seek( const int fd, const long long pos )
  {
  if( lseek( fd, pos, SEEK_SET ) == pos ) return true;
  show_error( "Seek error", errno ); return false;
  }


int list_files( const std::vector< std::string > & filenames,
                const int verbosity )
  {
  Pretty_print pp( filenames, verbosity );
  int retval = 0;
  for( unsigned i = 0; i < filenames.size(); ++i )
    {
    pp.set_name( filenames[i] );
    const int tmp = list_file( filenames[i].c_str(), pp );
    if( tmp > retval ) retval = tmp;
    }
  return retval;
  }


int range_decompress( const std::string & input_filename,
                      const std::string & default_output_filename,
                      Block range, const int verbosity, const bool force,
                      const bool ignore, const bool to_stdout )
  {
  struct stat in_stats;
  const int infd = open_instream( input_filename.c_str(), &in_stats, true, true );
  if( infd < 0 ) return 1;

  Pretty_print pp( input_filename, verbosity );
  const File_index file_index( infd );
  if( file_index.retval() != 0 )
    { pp( file_index.error().c_str() ); return file_index.retval(); }

  if( range.end() > file_index.data_end() )
    range.size( std::max( 0LL, file_index.data_end() - range.pos() ) );
  if( range.size() <= 0 )
    { if( verbosity >= 0 ) pp( "Nothing to do." ); return 0; }

  if( verbosity >= 1 )
    {
    if( verbosity >= 2 )
      std::fprintf( stderr, "Decompressed file size = %sB\n",
                    format_num( file_index.data_end() ) );
    std::fprintf( stderr, "Decompressing range %sB to %sB (%sBytes)\n",
                  format_num( range.pos() ),
                  format_num( range.pos() + range.size() ),
                  format_num( range.size() ) );
    }

  if( to_stdout || default_output_filename.empty() )
    outfd = STDOUT_FILENO;
  else
    {
    output_filename = default_output_filename;
    if( !open_outstream( force, false, false, false ) )
      { close( infd ); return 1; }
    }

  int retval = 0;
  for( long i = 0; i < file_index.members(); ++i )
    {
    const Block & db = file_index.dblock( i );
    if( range.overlaps( db ) )
      {
      if( verbosity >= 3 )
        std::fprintf( stderr, "Decompressing member %3ld\n", i + 1 );
      const long long outskip = std::max( 0LL, range.pos() - db.pos() );
      const long long outend = std::min( db.size(), range.end() - db.pos() );
      const long long mpos = file_index.mblock( i ).pos();
      if( !safe_seek( infd, mpos ) ) { retval = 1; break; }
      const int tmp = decompress_member( infd, pp, mpos, outskip, outend );
      if( tmp && ( tmp != 2 || !ignore ) )
        cleanup_and_fail( tmp );
      if( tmp > retval ) retval = tmp;
      pp.reset();
      }
    }
  close( infd );
  retval = std::max( retval, close_outstream( &in_stats ) );
  if( verbosity >= 2 && retval == 0 )
    std::fputs( "Byte range decompressed successfully.\n", stderr );
  return retval;
  }
